import { O as writable } from './index4-CjZb0xuH.js';

const defaultState = {
  department: "",
  brand: "",
  currentStep: 1,
  questionIndex: 0,
  totalQuestions: [],
  completedQuestions: {},
  lastFeedbackDate: null,
  roundsPlayed: 0,
  isComplete: false,
  progress: 0,
  currentRoundStarted: false,
  currentRound: 1,
  answers: {}
};
const initialState = {
  roundsPlayed: 0,
  isComplete: false,
  progress: 0,
  currentRoundStarted: false,
  currentRound: 1,
  answers: {},
  feedbackShownForRounds: {},
  // Track which rounds we've shown feedback for
  lastFeedbackDate: null
  // Track when feedback was last shown
};
const createFeedbackStore = () => {
  const initialStateWithStorage = initialState;
  const { subscribe, set, update } = writable({ ...defaultState, ...initialStateWithStorage });
  return {
    subscribe,
    updateState: (newState) => update((state) => {
      const updatedState = { ...state, ...newState };
      return updatedState;
    }),
    startNewRound: () => {
      update((state) => {
        const updatedState = {
          ...state,
          currentRoundStarted: true
        };
        return updatedState;
      });
    },
    completeRound: () => {
      update((state) => {
        const updatedState = {
          ...state,
          roundsPlayed: state.roundsPlayed + 1,
          currentRoundStarted: false,
          currentRound: state.currentRound + 1,
          lastFeedbackDate: (/* @__PURE__ */ new Date()).toISOString()
          // Update last feedback date
        };
        return updatedState;
      });
    },
    completeQuestion: (question, answer) => {
      update((state) => {
        const newCompletedQuestions = { ...state.completedQuestions, [question]: answer };
        const totalAnswered = Object.keys(newCompletedQuestions).length;
        const totalQuestions = state.totalQuestions.length + 2;
        const progress = Math.round(totalAnswered / totalQuestions * 100);
        const updatedState = {
          ...state,
          completedQuestions: newCompletedQuestions,
          progress,
          questionIndex: state.questionIndex + (question === "department" || question === "brand" ? 0 : 1),
          isComplete: totalAnswered === totalQuestions,
          answers: { ...state.answers, [question]: answer }
        };
        return updatedState;
      });
    },
    setTotalQuestions: (questions) => {
      update((state) => {
        const updatedState = { ...state, totalQuestions: questions };
        return updatedState;
      });
    },
    reset: () => {
      const newState = { ...defaultState, ...initialState };
      set(newState);
    },
    shouldShowFeedback: () => {
      let shouldShow = false;
      update((state) => {
        if (!state.lastFeedbackDate) {
          shouldShow = true;
          return state;
        }
        const lastDate = new Date(state.lastFeedbackDate);
        const today = /* @__PURE__ */ new Date();
        shouldShow = lastDate.getDate() !== today.getDate() || lastDate.getMonth() !== today.getMonth() || lastDate.getFullYear() !== today.getFullYear();
        return state;
      });
      return shouldShow;
    },
    getCurrentQuestions: () => {
      let currentQuestions = [];
      update((state) => {
        if (state.totalQuestions.length > 0) {
          const startIdx = state.questionIndex;
          const endIdx = Math.min(startIdx + 2, state.totalQuestions.length);
          currentQuestions = state.totalQuestions.slice(startIdx, endIdx).filter((q) => !state.completedQuestions[q]);
        }
        return state;
      });
      return currentQuestions;
    },
    getProgress: () => {
      let progress = 0;
      update((state) => {
        const totalAnswered = Object.keys(state.completedQuestions).length;
        const totalQuestions = state.totalQuestions.length + 2;
        progress = Math.round(totalAnswered / totalQuestions * 100);
        return state;
      });
      return progress;
    },
    getCurrentRound: () => {
      let currentState;
      subscribe((store) => {
        currentState = store;
      })();
      return currentState.currentRound;
    },
    setRound: (round) => update((store) => {
      store.currentRound = round;
      console.log("🔄 Feedback Store: Round set to", store.currentRound);
      return store;
    }),
    saveAnswer: (questionId, answer) => update((store) => {
      store.answers[questionId] = answer;
      return store;
    }),
    incrementRound: () => {
      update((state) => {
        const newState = {
          ...state,
          currentRound: state.currentRound + 1,
          roundsPlayed: state.roundsPlayed + 1
        };
        console.log("🔄 Feedback Store: Incrementing round", {
          oldRound: state.currentRound,
          newRound: newState.currentRound
        });
        return newState;
      });
    },
    markFeedbackShown: (round) => {
      update((state) => {
        const newState = {
          ...state,
          feedbackShownForRounds: {
            ...state.feedbackShownForRounds,
            [round]: true
          }
        };
        console.log("📝 Feedback Store: Marked feedback shown for round", round);
        return newState;
      });
    },
    isFeedbackShown: (round) => {
      let currentState;
      subscribe((store) => {
        currentState = store;
      })();
      return !!currentState.feedbackShownForRounds[round];
    }
  };
};
const feedbackStore = createFeedbackStore();

export { feedbackStore as f };
//# sourceMappingURL=feedbackStore-SG6OrzjY.js.map
