import { writable } from 'svelte/store';
import { browser } from '$app/environment';

const STORAGE_KEY = 'feedback-store';

const defaultState = {
    department: '',
    brand: '',
    currentStep: 1,
    questionIndex: 0,
    totalQuestions: [],
    completedQuestions: {},
    lastFeedbackDate: null,
    roundsPlayed: 0,
    isComplete: false,
    progress: 0,
    currentRoundStarted: false,
    currentRound: 1,
    answers: {}
};

const initialState = {
    roundsPlayed: 0,
    isComplete: false,
    progress: 0,
    currentRoundStarted: false,
    currentRound: 1,
    answers: {},
    feedbackShownForRounds: {}, // Track which rounds we've shown feedback for
    lastFeedbackDate: null // Track when feedback was last shown
};

const createFeedbackStore = () => {
    // Load initial state from localStorage if available
    const storedState = browser && localStorage.getItem(STORAGE_KEY);
    const initialStateWithStorage = storedState ? JSON.parse(storedState) : initialState;

    const { subscribe, set, update } = writable({ ...defaultState, ...initialStateWithStorage });

    return {
        subscribe,
        updateState: (newState) => update(state => {
            const updatedState = { ...state, ...newState };
            if (browser) {
                localStorage.setItem(STORAGE_KEY, JSON.stringify(updatedState));
            }
            return updatedState;
        }),
        startNewRound: () => {
            update(state => {
                const updatedState = { 
                    ...state,
                    currentRoundStarted: true
                };
                if (browser) {
                    localStorage.setItem(STORAGE_KEY, JSON.stringify(updatedState));
                }
                return updatedState;
            });
        },
        completeRound: () => {
            update(state => {
                const updatedState = {
                    ...state,
                    roundsPlayed: state.roundsPlayed + 1,
                    currentRoundStarted: false,
                    currentRound: state.currentRound + 1,
                    lastFeedbackDate: new Date().toISOString() // Update last feedback date
                };
                if (browser) {
                    localStorage.setItem(STORAGE_KEY, JSON.stringify(updatedState));
                }
                return updatedState;
            });
        },
        completeQuestion: (question, answer) => {
            update(state => {
                const newCompletedQuestions = { ...state.completedQuestions, [question]: answer };
                const totalAnswered = Object.keys(newCompletedQuestions).length;
                const totalQuestions = state.totalQuestions.length + 2; // +2 for department and brand
                const progress = Math.round((totalAnswered / totalQuestions) * 100);
                
                const updatedState = {
                    ...state,
                    completedQuestions: newCompletedQuestions,
                    progress,
                    questionIndex: state.questionIndex + (question === 'department' || question === 'brand' ? 0 : 1),
                    isComplete: totalAnswered === totalQuestions,
                    answers: { ...state.answers, [question]: answer }
                };
                
                if (browser) {
                    localStorage.setItem(STORAGE_KEY, JSON.stringify(updatedState));
                }
                return updatedState;
            });
        },
        setTotalQuestions: (questions) => {
            update(state => {
                const updatedState = { ...state, totalQuestions: questions };
                if (browser) {
                    localStorage.setItem(STORAGE_KEY, JSON.stringify(updatedState));
                }
                return updatedState;
            });
        },
        reset: () => {
            const newState = { ...defaultState, ...initialState };
            if (browser) {
                localStorage.setItem(STORAGE_KEY, JSON.stringify(newState));
            }
            set(newState);
        },
        shouldShowFeedback: () => {
            let shouldShow = false;
            update(state => {
                if (!state.lastFeedbackDate) {
                    shouldShow = true;
                    return state;
                }

                const lastDate = new Date(state.lastFeedbackDate);
                const today = new Date();
                
                // Check if it's a different day
                shouldShow = lastDate.getDate() !== today.getDate() ||
                           lastDate.getMonth() !== today.getMonth() ||
                           lastDate.getFullYear() !== today.getFullYear();
                
                return state;
            });
            return shouldShow;
        },
        getCurrentQuestions: () => {
            let currentQuestions = [];
            update(state => {
                if (state.totalQuestions.length > 0) {
                    const startIdx = state.questionIndex;
                    const endIdx = Math.min(startIdx + 2, state.totalQuestions.length);
                    currentQuestions = state.totalQuestions
                        .slice(startIdx, endIdx)
                        .filter(q => !state.completedQuestions[q]);
                }
                return state;
            });
            return currentQuestions;
        },
        getProgress: () => {
            let progress = 0;
            update(state => {
                const totalAnswered = Object.keys(state.completedQuestions).length;
                const totalQuestions = state.totalQuestions.length + 2; // +2 for department and brand
                progress = Math.round((totalAnswered / totalQuestions) * 100);
                return state;
            });
            return progress;
        },
        getCurrentRound: () => {
            let currentState;
            subscribe(store => {
                currentState = store;
            })();
            return currentState.currentRound;
        },
        setRound: (round) => update(store => {
            store.currentRound = round;
            console.log('🔄 Feedback Store: Round set to', store.currentRound);
            if (browser) {
                localStorage.setItem(STORAGE_KEY, JSON.stringify(store));
            }
            return store;
        }),
        saveAnswer: (questionId, answer) => update(store => {
            store.answers[questionId] = answer;
            if (browser) {
                localStorage.setItem(STORAGE_KEY, JSON.stringify(store));
            }
            return store;
        }),
        incrementRound: () => {
            update(state => {
                const newState = {
                    ...state,
                    currentRound: state.currentRound + 1,
                    roundsPlayed: state.roundsPlayed + 1
                };
                console.log('🔄 Feedback Store: Incrementing round', {
                    oldRound: state.currentRound,
                    newRound: newState.currentRound
                });
                if (browser) {
                    localStorage.setItem(STORAGE_KEY, JSON.stringify(newState));
                }
                return newState;
            });
        },
        markFeedbackShown: (round) => {
            update(state => {
                const newState = {
                    ...state,
                    feedbackShownForRounds: {
                        ...state.feedbackShownForRounds,
                        [round]: true
                    }
                };
                console.log('📝 Feedback Store: Marked feedback shown for round', round);
                if (browser) {
                    localStorage.setItem(STORAGE_KEY, JSON.stringify(newState));
                }
                return newState;
            });
        },
        isFeedbackShown: (round) => {
            let currentState;
            subscribe(store => {
                currentState = store;
            })();
            return !!currentState.feedbackShownForRounds[round];
        },
    };
};

export const feedbackStore = createFeedbackStore();
