import { writable } from 'svelte/store';

let questions = [];
let totalQuestions = 0;
let ratingQuestions = [];
let commentQuestions = [];
let selectedDepartment;

// Initial state for forms
const initialForms = {
    sales: {
        questions: [],
        brands: []
    },
    service: {
        questions: [],
        brands: []
    }
};

function createFormEditorStore() {
    const { subscribe, set, update } = writable({
        currentDepartment: 'sales',
        forms: initialForms,
        history: []
    });

    // Function to set the current department
    function setDepartment(department) {
        console.log(
            '[FormEditorStore] Setting department:',
            department
        );
        fetchQuestions(department);
        update(state => ({
            ...state,
            currentDepartment: department // Update the current department in the state
        }));
    }


    async function fetchFormIds() {
        try {
            const response = await fetch('/api/forms');
            if (response.ok) {
                const forms = await response.json();
                formIdMapping = forms.reduce((acc, form) => {
                    acc[form.department] = form.id;
                    return acc;
                }, {});
                console.log('[FormEditorStore] Fetched form IDs:', formIdMapping);
            } else {
                console.error('[FormEditorStore] Error fetching form IDs:', response.statusText);
            }
        } catch (error) {
            console.error('[FormEditorStore] Error fetching form IDs:', error);
        }
    }

    fetchFormIds();

    let formIdMapping = {};

    // Function to load questions based on selected department
    const fetchQuestions = async (selectedDepartment) => {
        if (selectedDepartment) {
            const response = await fetch(`/api/questions?formId=${formIdMapping[selectedDepartment]}`);
            if (response.ok) {
                const allQuestions = await response.json();
                console.log('Fetched questions:', allQuestions); // Log the fetched questions to check the structure
                
                // Directly assign the questions array
                if (Array.isArray(allQuestions)) {
                    questions = allQuestions; // Store the questions directly
                    totalQuestions = questions.length;
                    console.log('📝 Total questions:', totalQuestions);
                    // Optionally, you can separate rating and comment questions if needed
                    ratingQuestions = allQuestions.filter(q => q.type === 'rating');
                    commentQuestions = allQuestions.filter(q => q.type === 'comment');
                } else {
                    console.error('Expected an array of questions, but got:', allQuestions);
                }
            } else {
                console.error('Error fetching questions:', response.statusText);
            }
        }
    };

    // Function to load brands
    const fetchBrands = async () => {
        const currentDepartment = initialForms.currentDepartment;
        try {
            const response = await fetch('/api/brands');
            if (response.ok) {
                const brands = await response.json();
                update(state => ({
                    ...state,
                    forms: {
                        ...state.forms,
                        [currentDepartment]: {
                            ...state.forms[currentDepartment],
                            brands: brands
                        }
                    }
                }));
                console.log('[FormEditorStore] Fetched brands:', brands);
            } else {
                console.error('[FormEditorStore] Error fetching brands:', response.statusText);
            }
        } catch (error) {
            console.error('[FormEditorStore] Error fetching brands:', error);
        }
    };

    fetchBrands(); // Fetch brands for the new department

    // Function to add a new question
    async function addQuestion(question) {
        try {
            const response = await fetch('/api/questions', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(question),
            });

            if (response.ok) {
                const newQuestion = await response.json();
                update(state => {
                    const currentQuestions = state.forms[state.currentDepartment].questions;
                    return {
                        ...state,
                        forms: {
                            ...state.forms,
                            [state.currentDepartment]: {
                                ...state.forms[state.currentDepartment],
                                questions: [...currentQuestions, newQuestion]
                            }
                        }
                    };
                });
                console.log('[FormEditorStore] Added question:', newQuestion);
            } else {
                console.error('[FormEditorStore] Error adding question:', response.statusText);
            }
        } catch (error) {
            console.error('[FormEditorStore] Error adding question:', error);
        }
    }

    // Function to delete a question
    async function deleteQuestion(questionId) {
        try {
            const response = await fetch(`/api/questions/${questionId}`, {
                method: 'DELETE',
            });

            if (response.ok) {
                update(state => {
                    const questions = state.forms[state.currentDepartment].questions.filter(q => q.id !== questionId);
                    return {
                        ...state,
                        forms: {
                            ...state.forms,
                            [state.currentDepartment]: {
                                ...state.forms[state.currentDepartment],
                                questions
                            }
                        }
                    };
                });
                console.log('[FormEditorStore] Deleted question:', questionId);
            } else {
                console.error('[FormEditorStore] Error deleting question:', response.statusText);
            }
        } catch (error) {
            console.error('[FormEditorStore] Error deleting question:', error);
        }
    }

    // Function to reorder questions
    async function reorderQuestions(questions) {
        try {
            const response = await fetch(`/api/questions/reorder`, {
                method: 'PATCH',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(questions),
            });

            if (!response.ok) {
                throw new Error('Failed to reorder questions');
            }

            update(state => ({
                ...state,
                forms: {
                    ...state.forms,
                    [state.currentDepartment]: {
                        ...state.forms[state.currentDepartment],
                        questions: questions
                    }
                }
            }));
            console.log('[FormEditorStore] Reordered questions:', questions);
        } catch (error) {
            console.error('[FormEditorStore] Error reordering questions:', error);
        }
    }

    // Function to update a question
    async function updateQuestion(question) {
        try {
            const response = await fetch(`/api/questions/${question.id}`, {
                method: 'PATCH',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(question),
            });

            if (!response.ok) {
                throw new Error('Failed to update question');
            }

            const updatedQuestion = await response.json();
            update(state => {
                const questions = state.forms[state.currentDepartment].questions.map(q =>
                    q.id === updatedQuestion.id ? updatedQuestion : q
                );
                return {
                    ...state,
                    forms: {
                        ...state.forms,
                        [state.currentDepartment]: {
                            ...state.forms[state.currentDepartment],
                            questions
                        }
                    }
                };
            });
            console.log('[FormEditorStore] Updated question:', updatedQuestion);
        } catch (error) {
            console.error('[FormEditorStore] Error updating question:', error);
        }
    }

    return {
        subscribe,
        setDepartment,
        fetchQuestions,
        fetchBrands,
        addQuestion,
        deleteQuestion,
        reorderQuestions,
        updateQuestion
    };
}

export const formEditorStore = createFormEditorStore();