import { writable } from 'svelte/store';


function createGameStore() {
    const { subscribe, set, update } = writable({
        username: '',
        totalScore: 0,
        gameHistory: [],
        stats: {
            gamesPlayed: 0,
            bestScores: {
                puzzle: 0,
                race: 0,
                pairs: 0,
                trivia: 0
            },
            lastPlayed: null,
            playStreak: 0
        }
    });

    function updatePlayStreak(store) {
        const now = new Date();
        const lastPlayed = store.stats.lastPlayed ? new Date(store.stats.lastPlayed) : null;
        
        if (!lastPlayed) {
            return 1;
        }

        const daysSinceLastPlay = Math.floor((now - lastPlayed) / (1000 * 60 * 60 * 24));
        
        if (daysSinceLastPlay === 1) {
            return store.stats.playStreak + 1;
        } else if (daysSinceLastPlay > 1) {
            return 1;
        }
        
        return store.stats.playStreak;
    }

    return {
        subscribe,
        setUsername: (username) => {
            const savedData = localStorage.getItem(`gameData_${username}`);
            if (savedData) {
                set(JSON.parse(savedData));
            } else {
                update(store => ({
                    ...store,
                    username,
                    totalScore: 0,
                    gameHistory: [],
                    stats: {
                        gamesPlayed: 0,
                        bestScores: {
                            puzzle: 0,
                            race: 0,
                            pairs: 0,
                            trivia: 0
                        },
                        lastPlayed: null,
                        playStreak: 0
                    }
                }));
            }
        },
        addScore: (game, score) => {
            update(store => {
                const now = new Date().toISOString();
                const newGameHistory = [...store.gameHistory, {
                    game,
                    score,
                    timestamp: now
                }];

                const newStats = {
                    ...store.stats,
                    gamesPlayed: store.stats.gamesPlayed + 1,
                    bestScores: {
                        ...store.stats.bestScores,
                        [game]: Math.max(store.stats.bestScores[game], score)
                    },
                    lastPlayed: now,
                    playStreak: updatePlayStreak(store)
                };

                const newTotalScore = store.totalScore + score;

                const newStore = {
                    ...store,
                    totalScore: newTotalScore,
                    gameHistory: newGameHistory,
                    stats: newStats
                };

                // Save to localStorage
                localStorage.setItem(`gameData_${store.username}`, JSON.stringify(newStore));

                return newStore;
            });
        },
        getStats: () => {
            let stats;
            update(store => {
                stats = store.stats;
                return store;
            });
            return stats;
        },
        getGameHistory: () => {
            let history;
            update(store => {
                history = store.gameHistory;
                return store;
            });
            return history;
        }
    };
}

export const gameStore = createGameStore();
