import { writable } from 'svelte/store';

const INACTIVITY_TIMEOUT = 180000; // 3 minutes in milliseconds

function createScreensaverStore() {
    const { subscribe, set } = writable(false);
    let timer;

    function resetTimer() {
        if (timer) clearTimeout(timer);
        set(false);
        timer = setTimeout(() => set(true), INACTIVITY_TIMEOUT);
    }

    function initialize() {
        // Reset timer on various user interactions
        const events = ['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart'];
        
        events.forEach(event => {
            window.addEventListener(event, resetTimer, { passive: true });
        });

        // Initial timer setup
        resetTimer();

        // Cleanup function
        return () => {
            if (timer) clearTimeout(timer);
            events.forEach(event => {
                window.removeEventListener(event, resetTimer);
            });
        };
    }

    return {
        subscribe,
        initialize,
        reset: resetTimer
    };
}

export const screensaver = createScreensaverStore();
